/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/


/* ---------------------------------------------------------------
 * File: xiocommo.c 
 *       I/O routines
 * -----------------------------------------------------------------*/

/* 
 * Abstract:
 * This file contains I/O routines for the C-API.  These routines are
 *
 * DRIVER(PORT) and OS INDEPENDENT,
 * BEST-CARD DEPENDENT.
 *
 * The main purposes of the functions in this file:
 * 1. to encapsulate the algorithms (only) so that all "ports" are handled
 *    in the same manner.
 * 2. to reduce the amount of code necessary for each port driver.
 *
 * Some assumptions are made about any port;
 * 1. Each port has a driver such that BestXBasicRead/Write can be called.
 * 2. The port has been "opened" (i.e. BestXOpen() called) before any routines
 *    here are called.
 * 3. Each port should provide the following primitive routines or use the gen.
 *    purpose ones here.  (substitute a 3-letter port abbreviation for ***);
 *
 *    BestX***RegwidthSet()
 *    BestX***DeviceConnect()
 *    BestX***CheckConnection()
 *    BestX***ReleaseConnection()
 *
 *    Not normally needed but available...
 *    BestX***OnReadError()
 *    BestX***OnWriteError()
 *
 * See additional notes in bx_io.h
 */


#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

#include <xtypedef.h>
#include <xutil.h>

#include <xiocommo.h>

#include <xio.h>
#include <xcmd.h>
#include <xpci.h>

#if !defined(AIX) && !defined(_AIX) /* Don't want epp, hif, usb, or serial */
#include <xepp.h>
#include <xhif.h>
#include <xusb.h>
#include <xserial.h>
#endif

#include <xsession.h>
#include <xaddrmap.h>
#include <xlasterr.h>

#include <xerrcapi.h>
#include <xregcons.h>
#include <xregx10.h>
#include <xregx11.h>

#include <timeout.h>

/* for debug tracing */
#undef BBC_PROGRESS

#if defined (BEST_DEBUG) && defined (BBC_PROGRESS)
# define DBG_CBX_PRINTF(msg) { \
    if (bx_handlearray[handle].cbx_printf) \
      bx_handlearray[handle].cbx_printf((msg));}
#else
# define DBG_CBX_PRINTF(msg)
#endif

#undef BBBC_PROGRESS

#if defined (BEST_DEBUG) && defined (BBBC_PROGRESS)
# define DBG_CBX_PRINTF(msg) { \
    if (bx_handlearray[handle].cbx_printf) \
      bx_handlearray[handle].cbx_printf((msg));}
#else
# define DBG_CBX_PRINTF(msg)
#endif


#define EIGHT_LA "<", "<", "<", "<", "<", "<", "<", "<"
#define EIGHT_RA ">", ">", ">", ">", ">", ">", ">", ">"              
               
static char    *cbx_read_string_array [BX_MAXHANDLES] = {
            EIGHT_LA, EIGHT_LA, EIGHT_LA, EIGHT_LA,
            EIGHT_LA, EIGHT_LA, EIGHT_LA, EIGHT_LA,
            EIGHT_LA, EIGHT_LA, EIGHT_LA, EIGHT_LA,
            EIGHT_LA, EIGHT_LA, EIGHT_LA, EIGHT_LA,
            EIGHT_LA, EIGHT_LA, EIGHT_LA, EIGHT_LA,
            EIGHT_LA, EIGHT_LA, EIGHT_LA, EIGHT_LA,
            EIGHT_LA, EIGHT_LA, EIGHT_LA, EIGHT_LA,
            EIGHT_LA, EIGHT_LA, EIGHT_LA, EIGHT_LA
        };

static char    *cbx_write_string_array [BX_MAXHANDLES] = {
            EIGHT_RA, EIGHT_RA, EIGHT_RA, EIGHT_RA,
            EIGHT_RA, EIGHT_RA, EIGHT_RA, EIGHT_RA,
            EIGHT_RA, EIGHT_RA, EIGHT_RA, EIGHT_RA,
            EIGHT_RA, EIGHT_RA, EIGHT_RA, EIGHT_RA,
            EIGHT_RA, EIGHT_RA, EIGHT_RA, EIGHT_RA,
            EIGHT_RA, EIGHT_RA, EIGHT_RA, EIGHT_RA,
            EIGHT_RA, EIGHT_RA, EIGHT_RA, EIGHT_RA,
            EIGHT_RA, EIGHT_RA, EIGHT_RA, EIGHT_RA
        };

#if BX_MAXHANDLES > 256
#error Initialization of cbx_xx_string_array invalidated by increase in BX_MAXHANDLES
#endif


extern bx_int32    bestx_capi_prop [BX_MAXHANDLES + 1][BX_CAPIPROP_SIZE];
/* --------------------------------------------------------------------------
 * STATIC FUNCTIONS
 * -------------------------------------------------------------------------- */

/* --------------------------------------------------------------------------
 * Some low-level drivers have a bus-width > 1 byte and thus can do their
 * own byte-ordering (see Notes on RegWidth above).
 * Thus the name of this function could also be
 * BestXIsPortBusWidthMoreThanEightBits() (but that's too long).
 * -------------------------------------------------------------------------- */

static int BestXIsIoByteOrdered(bx_handletype handle)
{
  switch (bx_handlearray[handle].port)
  {
    case BX_PORT_OFFLINE:
    case BX_PORT_RS232:
    case BX_PORT_PARALLEL:
      break;

#ifdef CUSTOM_OEM1
    case BX_PORT_OEM:
#endif
    case BX_PORT_PCI_CONF:
    case BX_PORT_PCI_IO:
      return 1;

    case BX_PORT_FASTHIF:
    case BX_PORT_USB:
      break;

    default:
      assert(0 && "Illegal port in call to BestXIsIoByteOrdered()"); /*lint !e506 */
      return 0;  /*lint !e527 .. unreachable */
  }    /*lint !e788 ... not all enums used */
  return 0;
}


/* --------------------------------------------------------------------------
 * Each port type MUST have an appropriate BestX***DeviceConnect().
 * -------------------------------------------------------------------------- */

static bx_errtype BestXDeviceConnect(bx_handletype handle)
{
  BX_TRY_VARS_NO_PROG;

  /* handle was already checked before */
  bx_portnumtype OsHandle = bx_handlearray[handle].portnumber;

  BX_TRY_BEGIN 
  {
    switch (bx_handlearray[handle].port) 
    {
      case BX_PORT_OFFLINE:
        bx_handlearray[handle].is_connected = 1;
        break;
#if !defined(AIX) && !defined(_AIX)
      case BX_PORT_RS232:
        BX_TRY (BestXSerDeviceConnect(OsHandle));
        break;

      case BX_PORT_PARALLEL:
        BX_TRY (BestXParDeviceConnect(OsHandle));
        break;

      case BX_PORT_FASTHIF:
        /* ignore return value, CZ */
        (void)BestXHIFDeviceConnect (OsHandle);
        break;

      case BX_PORT_USB:
         if (bx_handlearray[handle].entered_port<BX_USB_PORTNUM_OFFSET)
         {
          (void)BestXUSBFXDeviceConnect(OsHandle);
         }
         else
         {
          (void)BestXUSBDeviceConnect(OsHandle);
         }
        break;
  #ifdef CUSTOM_OEM1
      case BX_PORT_OEM:
  #endif
#endif
      case BX_PORT_PCI_CONF:
      case BX_PORT_PCI_IO:
        BX_TRY (BestXPCIDeviceConnect(OsHandle));
        break;

      default:
        BX_TRY_ERROR(BX_E_INVALID_CASE);
    }    /*lint !e788 ... not all enums used */
  }

  /* TODO: check out whether the catch can be eliminated */
  BX_TRY_CATCH 
  {
    BX_TRY_RET = BX_E_CANNOT_CONNECT;
    BestXLastErrorParamSet (handle, BX_ERRPAR_1, (bx_int32)bx_handlearray[handle].port);
  }
  
  return BX_TRY_RET;
}


/* --------------------------------------------------------------------------
 * Some ports need an ack from the BestX
 * -------------------------------------------------------------------------- */

static int BestXBasicNeedsAck(bx_handletype handle)
{
  switch (bx_handlearray[handle].port)
  {
    case BX_PORT_OFFLINE:
      break;

    case BX_PORT_RS232:
      return 1;

#ifdef CUSTOM_OEM1
    case BX_PORT_OEM:
#endif
    case BX_PORT_PARALLEL:                       
    case BX_PORT_PCI_CONF:
    case BX_PORT_PCI_IO:
    case BX_PORT_FASTHIF:
    case BX_PORT_USB:
      break;

    default:
      assert(0 && "Illegal port in call to BestXBasicNeedsAck()"); /*lint !e506 */
      return 0; /*lint !e527 .. unreachable */
  }    /*lint !e788 ... not all enums used */
  return 0;
}


/* --------------------------------------------------------------------------
 * returns BX_E_OK if port closed during the timeout period (in ms.)
 * -------------------------------------------------------------------------- */

bx_errtype BestXWaitForClose(bx_handletype handle, bx_int32 timeout)
{
  /*
    chris:
    returns BX_E_OK if port closed during the timeout period (in ms.)
    returns BX_E_TRANSF_CMD, if timeout occurred
  */

  bx_errtype err;
  BEST_TIMER tmrTotal;
  BestStartTimeout(timeout, &tmrTotal);

  /* chris:
     BestXIsDisconnected() will return with BX_E_NOT_CONNECTED,
     if no error occurred and we are disconnected (BX_E_OK else)
  */
  
  while (BX_E_OK == (err = BestXIsDisconnected(handle)))
  {
    /* either we are still connected or an error has occurred */
    if (BestIsTimeoutDone(&tmrTotal)) 
    {
      BestXLastErrorParamSet (handle, BX_ERRPAR_1, (bx_int32)BX_EERR_CAPI_TIMEOUT);
      BestXLastErrorParamStringSet(handle,"BestXWaitForClose: Timeout detected by CAPI.");
      return BX_E_TRANSF_CMD;
    }
  }

  if (BX_E_NOT_CONNECTED == err)
  {
    /* we usually go here */
    return BX_E_OK;
  }

  return err;
}

/* --------------------------------------------------------------------------
 * return the maximum date width (in bytes) of any port
 * TODO; make real
 * -------------------------------------------------------------------------- */

static bx_int8 BestXGetBlockRegWidth(bx_handletype handle)
{
  switch (bx_handlearray[handle].port)
  {
    case BX_PORT_OFFLINE:
    case BX_PORT_RS232:
    case BX_PORT_PARALLEL:
      break;

#ifdef CUSTOM_OEM1
    case BX_PORT_OEM:
#endif

    case BX_PORT_PCI_CONF:
    case BX_PORT_PCI_IO:
      return 4;

    case BX_PORT_FASTHIF:
    case BX_PORT_USB:                                                                   /*1*****How big is the REGWIDTH for USB*******/
      break;

    default:
      assert (0);  /*lint !e506 */
  }    /*lint !e788 ... not all enums used */

  return 1;                     /* 64K default */
}


/* --------------------------------------------------------------------------
 * 2925 and 2926 w. reg# < 0x0a only
 * -------------------------------------------------------------------------- */

/* TODO: should not be exported */

bx_errtype EXPORT BestXBasicBlockWrite(bx_handletype handle,
    bx_int32 regnum,
    bx_int8ptr bt_ptr,
    size_t num_of_bytes)
{
  if (BestX16BitRegisterFile(handle) &&
    (regnum > REG_COMMON_MAX) && !IN_OFFLINE_MODE)
  {
    BestXLastErrorParamSet(handle, BX_ERRPAR_1, regnum);
    BX_ERRETURN(BX_E_UNTRANSLATED_CMD);
  }
  else
  {
    bx_errtype err;
    bx_sizetype regsize;
    /* scr; sorry about this but the options were limited */
    regsize = bx_regsize[(size_t) regnum].CardModel;

    /* old protocol */
    err = BestXBasicCommand8Bit(handle, regnum, CMD_WRITE,
      bt_ptr, regsize, num_of_bytes);

    /* TODO: why should I disconnect? */
    /* as former BX_E_ERROR was altered to BX_E_TRANSD_CMD the */
    /* test here is probably for BX_E_TRANSF_CMD, but who knows ?*/
    if ( (err == BX_E_ERROR) || (err== BX_E_TRANSF_CMD))
    {
      (void) BestXDisconnect(handle);
    }

    BX_ERRETURN(err);
  }
}

/* TODO: should not be exported */
bx_errtype EXPORT BestXBasicBlockRead(bx_handletype handle,
    bx_int32 regnum,           /* must be <= 0xf, if not in Core mode */
    bx_int8ptr bt_ptr,
    bx_int32 num_of_bytes)
{
  if (BestX16BitRegisterFile(handle) &&
    (regnum > REG_COMMON_MAX) && !IN_OFFLINE_MODE)
  {
    /* We are not in core mode, but the register number exceeds 0xf */
    BestXLastErrorParamSet(handle, BX_ERRPAR_1, regnum);
    BX_ERRETURN(BX_E_UNTRANSLATED_CMD);
  }

  else
  {
    bx_errtype err;
    bx_sizetype regsize;
    regsize = bx_regsize[(size_t) regnum].CardModel;

    err = BestXBasicCommand8Bit(handle, regnum, CMD_READ,
      bt_ptr, regsize, num_of_bytes);

    /* as former BX_E_ERROR was altered to BX_E_TRANSD_CMD the */
    /* test here is probably for BX_E_TRANSF_CMD, but who knows ?*/
    if ( (err == BX_E_ERROR) || (err== BX_E_TRANSF_CMD))
    {
      (void) BestXDisconnect(handle);
    }

    BX_ERRETURN(err);
  }
}


/* BestXBasicCommand
   The procedure consists of the following steps:
     1. send command code
     2. send parameters
     3. receive result
     4. receive return code
   Each of these steps increments the progress indicator of the try mechanism.
   If step 2 or 3 is skipped there is a dummy progress increment. The catch
   handler then computes the error message according to the achieved progress.
   CZ */
bx_errtype EXPORT BestXBasicCommand(bx_handletype handle,
    bx_int16 cmdcode,
    bx_int8ptr inptr,  
    bx_int16 insize,    /* number of bytes to write */
    bx_int8ptr outptr,  
    bx_int16 * outsize) /* number of bytes read */
{
  BX_TRY_VARS;
  bx_int8 buf[2] = {0, 0};
  bx_int8 reply;
  bx_int32 i;

  /* outsize may be null pointer only if outptr is null pointer */
  assert(outsize || !outptr);

  /* Some parameter repair (just to be sure) */
  if (insize==0) inptr = NULL;
  if ((outsize) && (*outsize==0)) outptr = NULL;

  BX_TRY_BEGIN
  {
    BX_TRY_HANDLE;

    if (BestXIsOffline(handle))
    {
      /* Offline always OK */
      if (outsize && outptr)
      {
        /* will need to read some bytes */      
        if (*outsize != BX_LENGTH_NEGOTIATE)
        {
          /* Number of bytes to read is fix/known/coded by command;
             Set everything to zero (offline mode!) */
          for (i=0;i<*outsize;i++)
          {
            outptr[i]=0;
          }
        }
        else
        {
          /* Number of bytes to read is unknown */         
          *outsize = 1;
          *outptr = '\0';
        }
      }
      BX_ERRETURN(BX_E_OK);
    } /* if (offline) */
    else if (!BestXHasFirmware(handle))
    {
      BX_E_ERROR_MSG_SET("BestXBasicCommand: Called w/o Firmware");
      BX_TRY_FAIL(BX_E_ERROR);
    }

    /* We are online and have FW running */

    /* split command code in two bytes, second byte in first position */
    buf[1] = (bx_int8) ((cmdcode & 0xff00) >> 8);  /* group (first command byte cmd1); sent second */
    buf[0] = (bx_int8) (cmdcode & 0x00ff);         /* modifier/command (second command byte cmd2); sent first */

    /* send command code */
    BX_TRY_PROGRESS(BestXBasicByteStreamWrite(handle, buf, 2UL));
    DBG_CBX_PRINTF(("BBC: opcode\n"));

    /* The card will disconnect if it cannot understand the command sent. */
    BX_TRY(BestXCheckConnection(handle));

    if (inptr)
    {
      /* Will have to write some bytes to card */
      if (insize == BX_LENGTH_NEGOTIATE)
      {
        /* length is transmitted; contained in first two bytes of buffer */
        BX_TRY(BestXBasicByteStreamWrite(handle, inptr, 2UL));
        inptr = BestXStream2Word(&insize, inptr, 1UL);
      }

      /* transmit data */
      BX_TRY_PROGRESS(BestXBasicByteStreamWrite(handle, inptr, (bx_int32) insize));
      DBG_CBX_PRINTF(("BBC: parameters\n"));
    }
    else
    {
      /* no bytes to write to card */
      
      /* synchronize progress */
      BX_TRY_PROGRESS(BX_E_OK);
    }

    if (outptr)
    {
      /* Will have to read some bytes from card */
      assert(outsize);

      /* read ONLY if you have valid pointers */
      if (*outsize == BX_LENGTH_NEGOTIATE)
      {
        /* length is received from card */
        BX_TRY(BestXBasicByteStreamRead(handle, (bx_int8ptr) outsize, 2UL));
        (void) BestXStream2Word(outsize, (bx_int8ptr) outsize, 1UL);
      }

      /* receive data */
      BX_TRY_PROGRESS(BestXBasicByteStreamRead(handle, outptr, (bx_int32) * outsize));
      DBG_CBX_PRINTF(("BBC: result\n"));
    }
    else
    {
      /* synchronize progress */
      BX_TRY_PROGRESS(BX_E_OK);
    }

    /* receive status */
    BX_TRY_PROGRESS(BestXBasicByteRead(handle, &reply));
    DBG_CBX_PRINTF(("BBC: reply (%02x\\h)\n", (long) reply));

    BX_TRY_FAIL(reply ? (bx_errtype) reply + BX_E_FIRMWARE : BX_E_OK);
  }

  BX_TRY_CATCH
  {
    if (cmdcode == CMD_ERRPAR_GET)
    {
      /* TODO: find good error message */
      return BX_E_FUNC;
    }

    /* do not remove even empty try passed blocks, CZ */
    BX_TRY_PASSED
    {
      /* command bytes */

      if (BX_TRY_RET == BX_E_NOT_CONNECTED)
      {
        BX_TRY_RET = BX_E_CONNECTION_LOST_CMD;
        BestXLastErrorParamSet(handle, BX_ERRPAR_2, (bx_int32) buf[1]);
        BestXLastErrorParamSet(handle, BX_ERRPAR_3, (bx_int32) buf[0]);
      }
    }

    BX_TRY_PASSED
    {
      /* in data */

      if (BX_TRY_RET == BX_E_NOT_CONNECTED)
      {
        BX_TRY_RET = BX_E_CONNECTION_LOST;
      }
    }

    BX_TRY_PASSED
    {
      /* out data */
    }

    BX_TRY_PASSED
    {
      bx_errtype err = BX_E_FUNC;
      bx_int8 outbuf[OUT_ERRPAR_GET];
      bx_int32 errpar[BX_ERROR_NUMPARAMS];
      bx_int16 outputsize = OUT_ERRPAR_GET;
      /* return code */
      if (BX_TRY_RET == BX_EFW_LONG_OPERATION)
      {
        reply = (bx_int8) (BX_EFW_LONG_OPERATION - BX_E_FIRMWARE);

        while (reply == (bx_int8) (BX_EFW_LONG_OPERATION - BX_E_FIRMWARE))
        {
          err = BestXBasicByteRead(handle, &reply);

          if (bx_handlearray[handle].cbx_printf != NULL)
          {
            bx_handlearray[handle].cbx_printf(".");
          }

          if (err)
          {
            /* error in transmission */
            return err;
          }
        }

        BX_TRY_RET = (bx_errtype) reply;
      }

      if (BX_TRY_RET)
      {
        DBG_CBX_PRINTF(("\n"));

        err = BestXBasicCommand(handle, CMD_ERRPAR_GET,
          NULL, IN_ERRPAR_GET, outbuf, &outputsize);

        if (!err)
        {
          (void) BestXStream2Long(errpar, outbuf, (bx_int32) BX_ERROR_NUMPARAMS);
          BestXLastErrorAllParamsSet(handle,
            errpar[0], errpar[1], errpar[2],
            errpar[3], errpar[4]);
          BX_TRY_RET = (bx_errtype) reply + BX_E_FIRMWARE;
        }
      }
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}


/******************************************************************************/
/* organizes transfers of variable lenght */
bx_errtype EXPORT BestXBasicCommandVariable(bx_handletype handle,
    bx_int16 cmdcode,
    bx_int8ptr inptr,
    bx_int16 insize,
    bx_int8ptr outptr,
    bx_int16 * outsize)
{
  bx_int8ptr buf;
  bx_errtype err;
  if ((buf = (bx_int8ptr) BestXMemMalloc(insize + 2)) == NULL) /* create a new buffer */
  {
    return BX_E_HOST_MEM_FULL;
  }

  if (inptr)
  {
    BESTX_MEMCPY(buf + 2, inptr, insize); /* copy from old to new buffer */
  }

  /* put into buffer */
  (void) BestXWord2Stream(buf, &insize, 1UL);

  err = BestXBasicCommand(handle, cmdcode, buf,  /* call standard fct */
    BX_LENGTH_NEGOTIATE, outptr, outsize);

  BestXMemFree((void*)&buf);                    /* free the allocated memory */

  return err;
}

/* This functions implements the 8Bit protocol used to execute a command on the card  */ 

bx_errtype BestXBasicCommand8Bit(
    bx_handletype handle,
    bx_int32 addr,           /* 8-bit address to be sent to card; contains command       */
    bx_int8 cmd,             /* wether to read or write data (CMD_READ=0 or CMD_WRITE=1) */
    bx_int8ptr  byteptr,     /* data to be read/written                                  */
    bx_int8 chunksize,       /* used register width (1,2 or 4); 
                                i.e. the minimum number 
                                of bytes, which can be transferred                       */
    size_t NumBytes)         /* length of data at byteptr                                */
{
  BX_DECLARE_FUNCNAME ("BestXBasicCommand8Bit");
  BX_TRY_VARS_NO_PROG;
  bx_int8 OneByteBuff;
  bx_int8 NumChunksPerTransfer;
  size_t NumChunksLeft;
  char *cbx_string;

  BX_TRY_BEGIN
  {
    /* Note that we do run-time null ptr. checking here...below this level we'll only be asserting */
    BX_TRY_FAIL(NumBytes && (NULL == byteptr) ? BX_E_PARAM_NULL_POINTER : BX_E_OK);

    /* Check register width (must be 1,2 or 4) */
    assert(BestXIsRegWidthValid(chunksize) && "Invalid reg. width in BestXBasicCommand8Bit()"); 
    
    /* Check, wether NumBytes matches register width */
    assert(0 == (NumBytes % chunksize) && "Boundary violation in BestXBasicCommand8Bit()");

    /* Number of chunks to be transferred in total */
    NumChunksLeft = NumBytes / (bx_int32) chunksize;

    /* We can only transfer a maximum of 127 (MAX_BLOCK_LEN_2925)
       data chunks per transfer (protocol limitation).
       So we split up into a number of smaller transfers. 
    */ 
    while (NumChunksLeft > 0L) /* still chunks to transfer ? */
    {
      /* Execute a transfer of maximum 127 chunks */

      if (NumChunksLeft > MAX_BLOCK_LEN_2925)
      {
        /* Still a lot of data left to transfer, 
           so we transfer maximum possible number of chunks (127)
        */
        NumChunksPerTransfer = MAX_BLOCK_LEN_2925;
        NumChunksLeft -= (bx_int32) NumChunksPerTransfer;
      }
      else
      {
        /* Last transfer; contains between 1 and 127 chunks */
        NumChunksPerTransfer = (bx_int8) NumChunksLeft;
        NumChunksLeft = 0L;
      }

      /* Connection is checked before each cmd */
      BX_TRY(BestXCheckConnection(handle));

      /* Protocol for one transfer: 
         - To card: byte 0: cmd (bit 7) and number of chunks (bit 0-6) 
         - To card: byte 1: reg-no (action to perform)        
         - Data transfer: NumChunksPerTransfer * chunksize bytes are transferred
           The direction depends on bit 7 of byte 0 (0=read; 1=write) 
         - From card: One status byte
      */

      /* write cmd-bit and number of chunks */
      OneByteBuff = (bx_int8) ((NumChunksPerTransfer & 0x7f) | (cmd << 7));
      BX_TRY(BestXBasicByteWrite(handle, &OneByteBuff));

      /* write register number */ 
      OneByteBuff = (bx_int8) addr;
      BX_TRY(BestXBasicByteWrite(handle, &OneByteBuff));

      /* Only the serial port needs an ack HERE. */
      if (BestXBasicNeedsAck(handle))
      {
        BX_TRY(BestXBasicByteRead(handle, &OneByteBuff));
      }

      OneByteBuff = STATUS_OK;

      /* Data transfer */
      switch (cmd)
      {
        case CMD_READ:
          /* Read  NumChunksPerTransfer * chunksize bytes from card */
          BX_TRY(BestXBasicByteOrderRead(handle,
              byteptr,
              (bx_int32)NumChunksPerTransfer,
              chunksize));

          byteptr += (int) (NumChunksPerTransfer * chunksize);
          cbx_string = cbx_read_string_array [handle];
          break;

        case CMD_WRITE:
          /* Write  NumChunksPerTransfer * chunksize bytes to card */
          BX_TRY(BestXBasicByteOrderWrite(handle,
              byteptr,
              (bx_int32)NumChunksPerTransfer,
              chunksize));

          byteptr += (int) (NumChunksPerTransfer * chunksize);
          cbx_string = cbx_write_string_array [handle];
          break;

        default:
          BX_TRY_FCT_PARAM (cmd, (cmd != CMD_WRITE) && (cmd != CMD_READ));
      }

      if (addr==BOARD_RESET)
      {
        /* Card has been resetted (e.g. BiosSwitch()) 
           so we dont need to read the status
           (which usually timeouts with BX_E_TRANSF_CMD).
           This has one side-effect: This function 
           continues immediately, while card maybe still
           booting. This has been implemented because
           USB supports no timeouts up to now.
        */
        BX_TRY_FAIL(BX_E_TRANSF_CMD);
      }
      else
      {
        /* Read status from card */
        BX_TRY(BestXBasicByteRead(handle, &OneByteBuff));
      }
      
      /* call back */
      if (bx_handlearray[handle].cbx_printf != NULL)
      {
        bx_handlearray[handle].cbx_printf(cbx_string);
      }

      BX_TRY_FAIL(OneByteBuff ? OneByteBuff + BX_E_FIRMWARE : BX_E_OK);
      

      /* next block */
    }
  }

  BX_TRY_CATCH
  {
    if (BX_TRY_RET == BX_EFW_LONG_OPERATION)
    {
      OneByteBuff = BX_EFW_LONG_OPERATION - BX_E_FIRMWARE;

      while (OneByteBuff == BX_EFW_LONG_OPERATION - BX_E_FIRMWARE)
      {
        bx_errtype err = BX_E_FUNC;
        err = BestXBasicByteRead(handle, &OneByteBuff);

        if (bx_handlearray[handle].cbx_printf != NULL)
        {
          bx_handlearray[handle].cbx_printf(".");
        }

        if (err)
        {
          /* error in transmission */
          return err;
        }
      }

      BX_TRY_RET = OneByteBuff;
    }
  }

  return BX_TRY_RET;
}


/* --------------------------------------------------------------------------
 * The routine to be called for ANY port to open a connection...timeout in ms
 * -------------------------------------------------------------------------- */

bx_errtype BestXOpenConnection(bx_handletype handle, bx_int32 timeout)
{
  BX_TRY_VARS_NO_PROG;
  BEST_TIMER tmrTotal;

  BX_TRY_BEGIN 
  {
    /* try to connect */
    BX_TRY (BestXDeviceConnect(handle));
    /* start timeout */
    BestStartTimeout(timeout, &tmrTotal);

    /* check the validity of the connection. */
    /* if there's no timeout we'll loop forever (or until there's an error) */
    while (BestXCheckConnection(handle) != BX_E_OK)
    {
      if (BestIsTimeoutDone(&tmrTotal)) 
      {
        BestXLastErrorParamSet (handle, BX_ERRPAR_1,
                       (bx_int32)bx_handlearray[handle].port);
        BX_TRY_ERROR (BX_E_CANNOT_CONNECT);
      }
    }
  }
  
  return BX_TRY_RET;
}


/* --------------------------------------------------------------------------
 * See notes at the beginning of this file
 * -------------------------------------------------------------------------- */

bx_errtype BestXRegwidthSet(bx_handletype handle, bx_int8 RegWidth)
{
  bx_portnumtype OsHandle;
  assert(BestXIsRegWidthValid(RegWidth));

  /* get out if there's no change from last time */
  if (RegWidth == bx_handlearray[handle].regwidth)
  {
    return BX_E_OK;
  }

  /* set new value */
  bx_handlearray[handle].regwidth = RegWidth;

  OsHandle = bx_handlearray[handle].portnumber;

  switch (bx_handlearray[handle].port)
  {
    case BX_PORT_OFFLINE:
    case BX_PORT_RS232:
    case BX_PORT_PARALLEL:         /* SCR; 27.11.97; all par. drivers byte-aware
                                   * only */
      break;

#ifdef CUSTOM_OEM1
    case BX_PORT_OEM:
#endif
    case BX_PORT_PCI_CONF:
    case BX_PORT_PCI_IO:
      return BestXPCISetRegwidth(OsHandle, RegWidth);

    case BX_PORT_FASTHIF:
      /* SCR; 20.11.97; driver is byte-aware only */
    case BX_PORT_USB:
      break;

    default:
      return BX_E_INVALID_CASE;
  }  /*lint !e788 */

  assert((RegWidth == 1)
    && "Only byte-streams call default BestXRegwidthSet()");  /*lint !e506 */

  return BX_E_OK;
}


/* --------------------------------------------------------------------------
 * 2 different means of "checking" the connection...
 * a. Hardware handshake (DTR/DSR etc.)
 * b. PCI Config. Register
 * -------------------------------------------------------------------------- */

bx_errtype BestXCheckConnection(bx_handletype handle)
{
  BX_TRY_VARS_NO_PROG;
  
  bx_portnumtype OsHandle = bx_handlearray[handle].portnumber;

  BX_TRY_BEGIN 
  {
    switch (bx_handlearray[handle].port) 
    {
      case BX_PORT_OFFLINE:
        BX_TRY_FAIL (BestXIsHandleConnected(handle) ? BX_E_OK : BX_E_NOT_CONNECTED);
        break;
      
#if !defined(AIX) && !defined(_AIX)
      case BX_PORT_RS232:
        BX_TRY (BestXSerCheckConnection(OsHandle));
        break;
      
      case BX_PORT_PARALLEL:
        BX_TRY (BestXParCheckConnection(OsHandle));
        break;
      
      case BX_PORT_FASTHIF:
        BX_TRY (BestXHIFCheckConnection(OsHandle));
        break;

      case BX_PORT_USB:
         if (bx_handlearray[handle].entered_port<BX_USB_PORTNUM_OFFSET)
         {
           BX_TRY (BestXUSBFXCheckConnection(OsHandle));
         }
         else
         {
           BX_TRY (BestXUSBCheckConnection(OsHandle));
         }
         break;
#ifdef CUSTOM_OEM1
      case BX_PORT_OEM:
#endif
#endif      

      case BX_PORT_PCI_CONF:
      case BX_PORT_PCI_IO:
        BX_TRY (BestXPCICheckConnection(OsHandle));
        break;

      default:
        BX_TRY_ERROR(BX_E_INVALID_CASE);
    }  /*lint !e788 */
  }

  BX_TRY_CATCH 
  {
    BX_TRY_RET = BX_E_NOT_CONNECTED;
    BestXLastErrorParamSet (handle, BX_ERRPAR_1, (bx_int32)bx_handlearray[handle].port);
  }
  
  return BX_TRY_RET;
}


bx_errtype BestXIsDisconnected (bx_handletype handle)
{
  BX_TRY_VARS_NO_PROG;
  
  bx_portnumtype OsHandle = bx_handlearray[handle].portnumber;

  BX_TRY_BEGIN 
  {
    switch (bx_handlearray[handle].port) 
    {
      case BX_PORT_OFFLINE:
        BX_TRY_FAIL (BestXIsHandleConnected(handle) ? BX_E_OK : BX_E_NOT_CONNECTED);
        break;
      
#if !defined(AIX) && !defined(_AIX)      
      case BX_PORT_RS232:
        BX_TRY (BestXSerCheckConnection(OsHandle));
        break;
      
      case BX_PORT_PARALLEL:
        BX_TRY (BestXParCheckConnection(OsHandle));
        break;
      
      case BX_PORT_FASTHIF:
        BX_TRY (BestXHIFIsDisconnected(OsHandle));
        break;

      case BX_PORT_USB:
         if (bx_handlearray[handle].entered_port<BX_USB_PORTNUM_OFFSET)
         {
           BX_TRY (BestXUSBFXCheckConnection(OsHandle));
         }
         else
         {
           BX_TRY (BestXUSBCheckConnection(OsHandle));
         }
         break;
#ifdef CUSTOM_OEM1
      case BX_PORT_OEM:
#endif
#endif

      case BX_PORT_PCI_CONF:
      case BX_PORT_PCI_IO:
        BX_TRY(BestXPCICheckConnection(OsHandle));
        break;

      default:
        BX_TRY_ERROR(BX_E_INVALID_CASE);
    }  /*lint !e788 */
  }

  BX_TRY_CATCH 
  {
    BX_TRY_RET = BX_E_NOT_CONNECTED;
    BestXLastErrorParamSet (handle, BX_ERRPAR_1, (bx_int32)bx_handlearray[handle].port);
  }
  
  return BX_TRY_RET;
}


/* --------------------------------------------------------------------------
 *
 * -------------------------------------------------------------------------- */

void BestXReleaseConnection(bx_handletype handle)
{
  /* Disconnect is somehow not fully implemented for the E2925A
     under PCI connection.
     Do not wait for it. CZ */

  bx_portnumtype OsHandle = bx_handlearray[handle].portnumber;
  
  switch (bx_handlearray[handle].port) 
  {
    case BX_PORT_OFFLINE:
      bx_handlearray[handle].is_connected = 0;
      break;

#if !defined(AIX) && !defined(_AIX)
    case BX_PORT_RS232:
      BestXSerReleaseConnection(OsHandle);
      break;

    case BX_PORT_PARALLEL:
      BestXParReleaseConnection(OsHandle);
      break;

    case BX_PORT_FASTHIF:
      BestXHIFReleaseConnection(OsHandle);
      break;

    case BX_PORT_USB:
      if (bx_handlearray[handle].entered_port<BX_USB_PORTNUM_OFFSET)
      {
        BestXUSBFXReleaseConnection(OsHandle);
      }
      else
      {
        BestXUSBReleaseConnection(OsHandle);
      }
      break;

#ifdef CUSTOM_OEM1
    case BX_PORT_OEM:
#endif

#endif

    case BX_PORT_PCI_CONF:
    case BX_PORT_PCI_IO:
      BestXPCIReleaseConnection(OsHandle);
      break;

    case BX_PORT_PRODUCTION:
      break;

    default:
      assert(0 && "Bad port in handle array; BestXReleaseConnection()"); /*lint !e506 */
      break;
  }  /*lint !e788 */

  (void)BestXWaitForClose(handle, 1000UL);

  return;
}


/* --------------------------------------------------------------------------
 * These functions allow special actions or error messages following a
 * failure of BestXBasicRead() or BestXBasicWrite().
 * They should not normally need to be implemented.  Serial comm. is one
 * exception.
 * -------------------------------------------------------------------------- */

bx_errtype BestXOnReadError(bx_handletype handle)
{
  DBG_ApiLastError;

  switch (bx_handlearray[handle].port)
  {
    case BX_PORT_OFFLINE:
      break;

#if !defined(AIX) && !defined(_AIX)
    case BX_PORT_RS232:
      return BestXSerOnReadOrWriteError(bx_handlearray[handle].portnumber);
#endif

#ifdef CUSTOM_OEM1
    case BX_PORT_OEM:
#endif

    case BX_PORT_PARALLEL:
    case BX_PORT_PCI_CONF:
    case BX_PORT_PCI_IO:
    case BX_PORT_FASTHIF:
    case BX_PORT_USB:            /*1*****DO WE NEED ERROR MESSAGES BY FAILURE OF BestXBasicRead() or BestXBasicWrite().*/
      break;

    default:
      return BX_E_INVALID_CASE;  /*lint !e506 */
  }  /*lint !e788 */
  BestXLastErrorParamStringSet(handle,"Generic on read error");
  BestXLastErrorParamSet (handle, BX_ERRPAR_1, (bx_int32)BX_EERR_GENERIC_READ);
  return BX_E_TRANSF_CMD;
}

bx_errtype BestXOnWriteError(bx_handletype handle)
{
  DBG_ApiLastError;

  switch (bx_handlearray[handle].port)
  {
    case BX_PORT_OFFLINE:
      break;

#if !defined(AIX) && !defined(_AIX)
    case BX_PORT_RS232:
      return BestXSerOnReadOrWriteError(bx_handlearray[handle].portnumber);
#endif

#ifdef CUSTOM_OEM1
    case BX_PORT_OEM:
#endif
    case BX_PORT_PARALLEL:
    case BX_PORT_PCI_CONF:
    case BX_PORT_PCI_IO:
    case BX_PORT_FASTHIF:
    case BX_PORT_USB:            /*1*****DO WE NEED ERROR MESSAGES BY FAILURE OF BestXBasicRead() or BestXBasicWrite().*/
      break;

    default:
    return BX_E_INVALID_CASE;  /*lint !e506 */
  }  /*lint !e788 */
  
  BestXLastErrorParamStringSet(handle,"Generic on write error."); 
  BestXLastErrorParamSet (handle, BX_ERRPAR_1, (bx_int32)BX_EERR_GENERIC_WRITE);
  return BX_E_TRANSF_CMD;
}


/* --------------------------------------------------------------------------
 * This function allows any driver to check for a user's callback function
 * during read/write.  It's called from BestXBasicRead().
 * Returns an error code (...stop processing on error) or BX_E_OK
 * -------------------------------------------------------------------------- */

bx_errtype BestXCheckCallBack(bx_handletype handle)
{
  if (bx_handlearray[handle].cbx_getevent &&
    bx_handlearray[handle].cbx_getevent() != 0)
  {
    BestXLastErrorParamStringSet(handle,"The card returned an unknown reply byte");
    BestXLastErrorParamSet(handle, BX_ERRPAR_1, (bx_int32)BX_EERR_UNKNOWN);
    return BX_E_TRANSF_CMD;
  }

  return BX_E_OK;
}


/* --------------------------------------------------------------------------
 * Determination of 'endianess'.  Note; This is the "real" answer.
 * To save time it can be defined true (i.e. Intel only) in bx_io.h but
 * this function is pretty quick.
 * !!! THUS; DO NOT CALL THIS FUNCTION DIRECTLY !!!
 * -------------------------------------------------------------------------- */

static unsigned char chEndian[] = {0x12,0x34,0x56,0x78};

int BestXIsCurMachineLittleEndian(void)
{
  return (*(unsigned long *)chEndian == 0x78563412UL);
}
/* --------------------------------------------------------------------------
 * This function will read a stream from the current port and change the
 * byte order as necessary (depending on datatype/regwidth).
 * See bx_io.h for additional macros which use this function.
 * -------------------------------------------------------------------------- */

bx_errtype BestXBasicByteOrderRead(
    bx_handletype handle,
    bx_int8ptr  pData,          /* caller's buffer */
    bx_int32 NumChunks,         /* number of data elements to read (NOT bytes) */
    bx_int8 chunksize)          /* data element width in bytes */
{
  bx_errtype err;
  int fIsIoByteOrdered = BestXIsIoByteOrdered(handle);
  /* if the low level driver can only handle streams then we'll read using a
   * "fake" data-element width ... but that doesn't change our byte-ordering
   * requirements later on !! */

  bx_int8 StreamWidth = fIsIoByteOrdered ? chunksize : 1;
  if (NULL == pData)
  {
    return BX_E_PARAM_NULL_POINTER;
  }

  /* first "dump" the number of bytes from the port into the user's buffer */
  if (BX_E_OK != (err = BestXBasicRead(handle,
        pData,
        NumChunks * chunksize,
        StreamWidth))
    )
  {
    return err;
  }

  /* do we need to change the byte order ? */
  if (fIsIoByteOrdered || (chunksize == 1) || !BestXIsHostLittleEndian())
  {
    return BX_E_OK;
  }

  /* OH, DARN...we've got work to do. The cast to void * gets rid of
   * misalignment issues on different platforms */
  switch (chunksize)
  {
    case 2:
      (void) BestXStream2Word((bx_int16 *) (void *) pData, pData, NumChunks);
      break;

    case 4:
      (void) BestXStream2Long((bx_int32 *) (void *) pData, pData, NumChunks);
      break;

    default:
      assert(0
      && "Illegal data type (register width) in BestXBasicByteOrderRead");  /*lint !e506 */
      break;
  }
  return BX_E_OK;
}

#define BBBOW_BUF_SIZE  12


/* --------------------------------------------------------------------------
 * This function will write the caller's buffer "byte-ordered" to a port.
 * It malloc's buffers that are over BBBOW_BUF_SIZE bytes long
 * -------------------------------------------------------------------------- */

bx_errtype BestXBasicByteOrderWrite(
    bx_handletype handle,
    bx_int8ptr  pData,        /* caller's buffer */
    bx_int32 NumChunks,       /* number of data elements to write (NOT bytes) */
    bx_int8 chunksize)        /* data element width in bytes */
{
  bx_int8 BoBuff[BBBOW_BUF_SIZE];
  bx_errtype err;
  bx_int8 fMallocUsed = 0;
  bx_int8ptr pOurBuff = pData;
  bx_int32 NumBytes = (NumChunks * chunksize);
  if (NULL == pData)
  {
    return BX_E_PARAM_NULL_POINTER;
  }

  /* do we need to change the order or will the driver do it? OR did we get
   * lucky and get a byte stream anyway? */
  if (!BestXIsIoByteOrdered(handle) && (chunksize > (bx_int8) 1))
  {
    /* we only need to change byte order for little-endian hosts but we do
     * need to change the data-element width...regardless */
    if (BestXIsHostLittleEndian())
    {
      /* do we need to malloc ? */
      if (NumBytes > sizeof(BoBuff))
      {
        if (NULL == (pOurBuff = (bx_int8ptr ) BestXMemMalloc((size_t) NumBytes)))
        {
          return BX_E_HOST_MEM_FULL;
        }
        fMallocUsed = 1;
      }
      else
      {
        pOurBuff = BoBuff;
      }

      /* do the conversion. The cast to void * gets rid of misalignment
       * issues on different platforms */
      switch (chunksize)
      {
        case 2:
          (void) BestXWord2Stream(pOurBuff, (bx_int16 *) (void *) pData, NumChunks);
          break;

        case 4:
          (void) BestXLong2Stream(pOurBuff, (bx_int32 *) (void *) pData, NumChunks);
          break;

        default:
          assert(0
          && "Illegal data type (register width) in BestXBasicByteOrderRead");  /*lint !e506 */
          break;
      }
    }

    /* after conversion to a stream the data element width is 1 byte */
    chunksize = 1;
  }

  err = BestXBasicWrite(handle, pOurBuff, NumBytes, chunksize);

  if (fMallocUsed)
  {
    BestXMemFree((void*)&pOurBuff);
  }
  return err;
}
/* Default timeouts used for each port.
 * Note that it is only necessary to set the TIMEOUTS_DEF_xxxx in the
 * header file.  No other action need be taken by the port.
 * This default value is set each time the port is opened.
 * Kernel mode drivers need to set their own defaults if they read/write
 * as part of an open (or read/write without opening...mailbox).
 */

static BESTTIMEOUTS DefaultSerTimeouts = TIMEOUTS_DEF_SERIAL; 
static BESTTIMEOUTS DefaultParTimeouts = TIMEOUTS_DEF_PAR;
static BESTTIMEOUTS DefaultPciTimeouts = TIMEOUTS_DEF_PCI;
static BESTTIMEOUTS DefaultHifTimeouts = TIMEOUTS_DEF_FHIF;
static BESTTIMEOUTS DefaultUSBTimeouts = TIMEOUTS_DEF_USB;

static BESTTIMEOUTS NoTimeouts = NO_TIMEOUTS;
static BESTTIMEOUTS ImmReadTimeout = IMM_READ_TIMEOUT;
static BESTTIMEOUTS ImmWriteTimeout = IMM_WRITE_TIMEOUT;
static BESTTIMEOUTS ImmIoTimeouts = IMM_IO_TIMEOUTS;

/* --------------------------------------------------------------------------
 * Sets port timeouts.
 * note that we store the timeouts in the handle array
 * ... some multi-device drivers can't easily provide their own storage
 *
 * Passing
 *    pCallersTimeouts == NULL OR fSetAction = TIMEOUTS_SET_DEF
 *      will set the DEFAULT timeouts.
 *    fSetAction = TIMEOUTS_SET_NONE will set no timeouts (i.e. infinite).
 * -------------------------------------------------------------------------- */

bx_errtype EXPORT BestXPortTimeoutSet(
                        bx_handletype handle,
                        BESTTIMEOUTS * pCallersTimeouts,
                        int fSetAction)
{
  /* our own pointer */
  BESTTIMEOUTS *pHandleTimeouts = &(bx_handlearray[handle].timeouts);
  bx_portnumtype OsHandle = bx_handlearray[handle].portnumber;

  /* as the defaults are port-dependent we handle those differently */
  int fSetDefault = ((NULL == pCallersTimeouts) && (TIMEOUTS_SET_DEF == fSetAction));

  if (!fSetDefault)
  {
    switch (fSetAction)
    {
      case TIMEOUTS_SET:
        if(NULL == pCallersTimeouts)
        {
          return BX_E_PARAM_NULL_POINTER;
        }
        *pHandleTimeouts = *pCallersTimeouts;
        break;

      case TIMEOUTS_SET_NONE:
        *pHandleTimeouts = NoTimeouts;
        break;

      case TIMEOUTS_SET_IMM_READ:
        *pHandleTimeouts = ImmReadTimeout;
        break;

      case TIMEOUTS_SET_IMM_WRITE:
        *pHandleTimeouts = ImmWriteTimeout;
        break;

      case TIMEOUTS_SET_IMM_IO:
        *pHandleTimeouts = ImmIoTimeouts;
        break;

      default:
        return BX_E_INVALID_CASE;
    }
  }

  /* note that we're only calling the BestXxxxPortTimeoutSet() in case the
   * driver needs its own storage for the timeouts struct. */

  switch (bx_handlearray[handle].port)
  {
    case BX_PORT_OFFLINE:          /* use serial as the default */
      if (fSetDefault)
      {
        *pHandleTimeouts = DefaultSerTimeouts;
        break;
      }

#if !defined(AIX) && !defined(_AIX)
    case BX_PORT_RS232:
      if (fSetDefault)
      {
        *pHandleTimeouts = DefaultSerTimeouts;
      }
      return BestXSerPortTimeoutSet(OsHandle, pHandleTimeouts);

    case BX_PORT_PARALLEL:
      if (fSetDefault)
      {
        *pHandleTimeouts = DefaultParTimeouts;
      }
      return BestXParPortTimeoutSet(OsHandle, pHandleTimeouts);

    case BX_PORT_FASTHIF:
      if (fSetDefault)
      {
        *pHandleTimeouts = DefaultHifTimeouts;
      }
      return BestXHifPortTimeoutSet(OsHandle, pHandleTimeouts);

    case BX_PORT_USB:
      if (fSetDefault)
      {
        *pHandleTimeouts = DefaultUSBTimeouts;
      }
      return BestXUSBPortTimeoutSet(OsHandle, pHandleTimeouts);  
#ifdef CUSTOM_OEM1
    case BX_PORT_OEM:
#endif
#endif

    case BX_PORT_PCI_CONF:
    case BX_PORT_PCI_IO:
#if 1
      if (fSetDefault)
      {
        *pHandleTimeouts = DefaultPciTimeouts;
      }
      return BestXPciPortTimeoutSet(OsHandle, pHandleTimeouts);
#else
      return BX_E_OK;
#endif

    default:

    case BX_PORT_PRODUCTION:
      return BX_E_OK;

      
      return BX_E_INVALID_CASE;
  } /*lint !e788 */
  return BX_E_OK;
}


/* --------------------------------------------------------------------------
 * Block mode I/O mechanisms vary between ports...here we take care of
 * those differences and establish common algorithms.
 * NOTE; ALL block mode I/O is streamed !!
 *       Byte-ordering is NOT performed and driver access is 8-bit.
 *       Granularity is the boundary width of an I/O request and can
 *       be any number of bytes.  It ensures that an I/O packet will begin AND
 *       end on the requested boundary.
 *       This could also be thought of as "Sentence Width".
 *       MaxBlockSize is port-dependent.
 * -------------------------------------------------------------------------- */

/* --------------------------------------------------------------------------
 * Put the card into block (direct) mode
 * -------------------------------------------------------------------------- */

static bx_errtype BestXBlockIoSetup(
    bx_handletype handle,
    bx_int32 address,           /* direct address on board */
    bx_int32 numbytes,          /* length of data at byteptr */
    bx_int8 regwidth,
    bx_int8 writecmd,           /* CMD_READ or CMD_WRITE */
    bx_int8 autoinc,            /* onboard resource to read/write */
    bx_int32 * pBlockBytes      /* # bytes the port transfers per access */
)
{
  bx_errtype err = BX_E_OK;
  bx_int8 buf[IN_BLOCK_MODE_SET + 1];
  bx_int8ptr bp = buf;
  bx_bool directMode;

  /* firmware relies on this in block mode set up. CZ */
  assert (CMD_READ == 0x00);  /*lint !e506 */
  assert (CMD_WRITE == 0x01);  /*lint !e506 */
  assert (writecmd <= CMD_WRITE);
  
  switch (bx_handlearray[handle].port) 
  {
   case BX_PORT_USB:
    if (BestXHasFaust(handle) && writecmd==CMD_READ && address==BX_REG_DT_DATAREAD_F)
    {
      /* Its a tracememory read from a Faust card; use special fast mode */
      if (BestXIsMode2(handle))
      {
        writecmd |= 1<<6; /* needed to determine width of tracememory */
      }
    }
    *pBlockBytes = USB_BLOCKSIZE_MAX; 

    /* indication to actually use the fast mode */
    /* Direct, HW supported mode */
    writecmd |= 1<<7;
    directMode = BX_TRUE;
    break;

   case BX_PORT_FASTHIF:
    *pBlockBytes = FHIF_BLOCKSIZE_MAX; /* 0x1eff0=124KB-16B */ 
    /* indication to actually use the fast mode */
    /* Direct, HW supported mode */
    writecmd |= 1<<7;
    directMode = BX_TRUE;
    break;

   default:
    *pBlockBytes = numbytes; /* whatever */
    /* nothing to do */
    break;
  }  /*lint !e788 */
  
  bp = BestXLong2Stream(bp, &address, 1UL);
  bp = BestXLong2Stream(bp, &numbytes, 1UL);

  bp = BestXByteCopy (bp, &regwidth, 1UL);
  bp = BestXByteCopy (bp, &writecmd, 1UL);
  bp = BestXByteCopy (bp, &autoinc, 1UL);
  *bp = 0;   /* what is that for? HL */

  err = BestXBasicCommand(handle, CMD_BLOCK_MODE_SET,
        buf, IN_BLOCK_MODE_SET,
        NULL, NULL);

  if (err == BX_E_OK)
  {
    BestXHwBlockModeBitSet(handle, 1);
    BestXHwDirectModeBitSet(handle, directMode);
  }
  
  return err;
}


/* --------------------------------------------------------------------------
 * Get the card out of block mode
 * -------------------------------------------------------------------------- */

static bx_errtype BestXBlockIoCleanUp(
    bx_handletype handle
)
{
  bx_errtype    err = BX_E_OK;

  BestXHwBlockModeBitSet(handle, BX_FALSE);
  BestXHwDirectModeBitSet(handle, BX_FALSE);

#if 0
  err = BestXBasicCommand (handle, CMD_BLOCK_MODE_CLEANUP,
              NULL, IN_BLOCK_MODE_CLEANUP,
              NULL, NULL);
#endif
  return err;
}
/* --------------------------------------------------------------------------
 * Block Reads and Writes are identically formed at this level.
 * -------------------------------------------------------------------------- */

static bx_errtype BestXBlockIo(
    bx_handletype handle,
    bx_int32 NumBytes,           /* number of bytes */
    int write_cmd,               /* CMD_READ or CMD_WRITE */
    bx_int8ptr  pData            /* caller's buffer */
)
{
  BX_TRY_VARS_NO_PROG;

  bx_int8 RegWidth = BestXGetBlockRegWidth(handle);
  
  BX_TRY_BEGIN 
  {
    /* the actual I/O ... note that BestXBasicRead/Write is block-mode aware */
    if (CMD_READ == write_cmd) 
    {
      BX_TRY(BestXBasicRead(handle, pData, NumBytes, RegWidth));
    }
    else 
    {
      BX_TRY(BestXBasicWrite(handle, pData, NumBytes, RegWidth));
    }
  }

  return BX_TRY_RET;
}


/* -------------------------------------------------------------------------
 * This is the interface for CAPI programs to perform block (direct) I/O
 * ------------------------------------------------------------------------- */

bx_errtype BestXBasicBlockCommand(
    bx_handletype handle,
    bx_int32 address,
    bx_int32 numbytes,
    bx_int8  regwidth,
    bx_int8  writecmd,   /* CMD_READ or CMD_WRITE */
    bx_bool  autoinc,    /* TRUE (HW increments) FALSE (SW increments)*/
    bx_int8ptr  pData    /* addr of caller's buffer to read/write */
)
{
  BX_DECLARE_FUNCNAME ("BestXBasicBlockCommand (not exported)");
  BX_TRY_VARS;

  /* This call sets up the card for block I/O and flags the handle to
   * indicate that we're in block mode...VERY important ! */

  BX_TRY_BEGIN 
  {
    bx_int32    blockBytes, transferBytes;
    
    /* parameter checking */
    BX_TRY_FCT_PARAM (regwidth, (regwidth != 2) && (regwidth != 4) );
    BX_TRY_FCT_PARAM_ALIGNMENT (address, regwidth);
    BX_TRY_FCT_PARAM_ALIGNMENT (numbytes, regwidth);

    /* prepare card for block IO:
       - Switch HW into blockmode (CMD_BLOCK_MODE_SET)
       - Set HWBITS BX_HWDEF_IS_BLKMODE and BX_HWDEF_IS_DIRECTMODE in handlearray 
       - Return blocksize (usually equals numbytes (FHIF: 124KB fix) */

    BX_TRY_PROGRESS ( BestXBlockIoSetup( handle,
                      address, numbytes,
                      regwidth, writecmd,
                      (bx_int8) (autoinc ? 1 : 0),
                      &blockBytes ) );

    DBG_CBX_PRINTF (("BBBC: setup\n"));
    
    /* -------------------- here comes the actual transfer! -----------------*/
    while (numbytes) 
    {
      transferBytes = min (numbytes,blockBytes); /* reduction to 124KB max. in case of FHIF */
    
      /* Do the actual I/O. Note that we must NEVER return without calling
       * BestXBlockIoCleanUp() ... that clears the handle's block I/O flag. */
      BX_TRY (BestXBlockIo(handle, transferBytes, writecmd, pData));
      DBG_CBX_PRINTF (("BBBC: block IO\n"));

      numbytes -= transferBytes;
      pData += transferBytes;
    }

    /* A USB write may add some bytes from the next (non-block) command
       which would arrive together with the last chunk of block
       data in the block ISR -> too many bytes are transferred 
       in block mode. So we wait a ms which leads to an block interrupt
       with the correct number of bytes */ 
    
    BESTX_SLEEP(16);

    BX_TRY_FAIL (BX_E_INTERNAL_RETURN); /* must be last line! */
  }
  
  BX_TRY_CATCH 
  {
    BX_TRY_PASSED 
    {
      /* BestXBlockIoSetup() was successful */
      bx_errtype err = BestXBlockIoCleanUp(handle);
      DBG_CBX_PRINTF (("BBBC: clean up\n"));

      if (BX_TRY_RET == BX_E_OK)
      {
        BX_TRY_RET = err;
      }
    }
    
    if (BX_TRY_RET == BX_E_INTERNAL_RETURN)
    {
      BX_TRY_RET = BX_E_OK;
    }
  }

  return BX_TRY_RET;
}


/* -------------------------------------------------------------------- */
/* control functions */

void BestXCBReadStringSet (bx_handletype handle, char *s)
{
  if (handle < BX_MAXHANDLES)
  {
    cbx_read_string_array [handle] = s;
  }
}


void BestXCBWriteStringSet (bx_handletype handle, char *s)
{
  if (handle < BX_MAXHANDLES)
  {
    cbx_write_string_array [handle] = s;
  }
}


